const Database = require('better-sqlite3');
const path = require('path');
const fs = require('fs');
const bcrypt = require('bcrypt');
const dbDir = path.join(__dirname, '..', 'databases');
if (!fs.existsSync(dbDir)) {
  fs.mkdirSync(dbDir);
}
const userDb = new Database(path.join(dbDir, 'main_app.db'));
console.log('✅ Connected to the main application database.');
const forwardingDb = new Database(path.join(dbDir, 'forwarding_data.db'));
console.log('✅ Connected to the forwarding notes database.');
const receiptsDb = new Database(path.join(dbDir, 'receipts_data.db'));
console.log('✅ Connected to the receipts database.');
const dispatchDb = new Database(path.join(dbDir, 'dispatch_data.db'));
console.log('✅ Connected to the dispatch orders database.');
const initializeSchema = () => {
  console.log('Initializing database schema...');
  userDb.exec(`
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            email TEXT NOT NULL UNIQUE,
            password_hash TEXT NOT NULL,
            role TEXT NOT NULL CHECK(role IN ('admin', 'user', 'manager'))
        );
        CREATE TABLE IF NOT EXISTS companies (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL UNIQUE
        );
        CREATE TABLE IF NOT EXISTS transporters (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL UNIQUE
        );
        CREATE TABLE IF NOT EXISTS user_companies (
            user_id INTEGER NOT NULL,
            company_id INTEGER NOT NULL,
            PRIMARY KEY (user_id, company_id),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE CASCADE
        );
        CREATE TABLE IF NOT EXISTS permissions (
            user_id INTEGER NOT NULL,
            permission TEXT NOT NULL,
            PRIMARY KEY (user_id, permission),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        );
    `);
  forwardingDb.exec(`
        CREATE TABLE IF NOT EXISTS notes (
            id TEXT PRIMARY KEY, sn TEXT NOT NULL UNIQUE, note_date TEXT NOT NULL, consignor TEXT NOT NULL, consignee TEXT,
            transport TEXT, destination TEXT, contents TEXT, quantity TEXT, value REAL, waybill TEXT, inv TEXT, freight TEXT,
            created_by_id INTEGER NOT NULL, created_by_name TEXT NOT NULL
        );
    `);
  receiptsDb.exec(`
        CREATE TABLE IF NOT EXISTS receipts (
            id TEXT PRIMARY KEY, receipt_no TEXT NOT NULL UNIQUE, dated TEXT NOT NULL, received_from TEXT, on_account_of TEXT,
            in_favour_of TEXT, amount REAL, sum_of_rupees TEXT, created_by_id INTEGER NOT NULL, created_by_name TEXT NOT NULL
        );
    `);

  // **FIXED**: Removed all FOREIGN KEY constraints that reference other database files.
  // Validation will now be handled in the application logic (routes.js).
  dispatchDb.exec(`
        CREATE TABLE IF NOT EXISTS orders (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            sr_no TEXT,
            party_name TEXT NOT NULL,
            place TEXT NOT NULL,
            district TEXT,
            quantity_kg REAL NOT NULL,
            company_id INTEGER NOT NULL,
            transporter_id INTEGER NOT NULL,
            status TEXT NOT NULL DEFAULT 'Open',
            is_urgent INTEGER NOT NULL DEFAULT 0,
            is_on_hold INTEGER NOT NULL DEFAULT 0,
            clubbed_with_id INTEGER,
            remark TEXT,
            hold_reason TEXT,
            created_by_id INTEGER NOT NULL,
            created_by_name TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );
    `);
  const adminExists = userDb.prepare('SELECT id FROM users WHERE role = ?').get('admin');
  if (!adminExists) {
    console.log('No admin user found. Creating default admin...');
    const adminPassword = 'adminpassword';
    const saltRounds = 10;
    const hash = bcrypt.hashSync(adminPassword, saltRounds);
    userDb.prepare('INSERT INTO users (name, email, password_hash, role) VALUES (?, ?, ?, ?)').run('Admin', 'admin@krishna.com', hash, 'admin');
    console.log('--------------------------------------------------');
    console.log('Default Admin Account Created:');
    console.log('Email: admin@krishna.com');
    console.log(`Password: ${adminPassword}`);
    console.log('--------------------------------------------------');
  }
  console.log('Schema initialization complete.');
};
initializeSchema();
module.exports = {
  userDb,
  forwardingDb,
  receiptsDb,
  dispatchDb
};