const express = require('express');
const router = express.Router();
const bcrypt = require('bcrypt');
const {
  userDb,
  forwardingDb,
  receiptsDb,
  dispatchDb
} = require('./database');
const {
  isAuthenticated,
  isAdmin,
  canDispatch,
  canUpdateDispatch
} = require('./auth');

// --- AUTHENTICATION & SESSION ---
router.post('/login', (req, res) => {
  const {
    email,
    password
  } = req.body;
  if (!email || !password) {
    return res.status(400).json({
      message: "Email and password are required."
    });
  }
  try {
    const user = userDb.prepare('SELECT * FROM users WHERE email = ?').get(email);
    if (!user || !bcrypt.compareSync(password, user.password_hash)) {
      return res.status(401).json({
        message: "Invalid credentials."
      });
    }
    const companies = userDb.prepare(`SELECT c.id, c.name FROM companies c JOIN user_companies uc ON c.id = uc.company_id WHERE uc.user_id = ?`).all(user.id);
    const permissions = userDb.prepare('SELECT permission FROM permissions WHERE user_id = ?').all(user.id).map(p => p.permission);
    req.session.user = {
      id: user.id,
      name: user.name,
      email: user.email,
      role: user.role,
      companies,
      permissions
    };
    res.status(200).json(req.session.user);
  } catch (err) {
    res.status(500).json({
      message: "Server error during login."
    });
  }
});
router.post('/logout', (req, res) => {
  req.session.destroy(() => {
    res.clearCookie('connect.sid');
    res.status(200).json({
      message: 'Logged out.'
    });
  });
});
router.get('/session/check', (req, res) => {
  res.status(req.session.user ? 200 : 401).json({
    user: req.session.user || null
  });
});

// --- ADMIN MANAGEMENT ---
router.get('/users', isAuthenticated, isAdmin, (req, res) => {
  try {
    const users = userDb.prepare('SELECT id, name, email, role FROM users').all();
    res.json(users.map(user => ({
      ...user,
      companies: userDb.prepare(`SELECT c.id, c.name FROM companies c JOIN user_companies uc ON c.id = uc.company_id WHERE uc.user_id = ?`).all(user.id),
      permissions: userDb.prepare('SELECT permission FROM permissions WHERE user_id = ?').all(user.id).map(p => p.permission)
    })));
  } catch (err) {
    res.status(500).json({
      message: err.message
    });
  }
});
router.post('/users', isAuthenticated, isAdmin, (req, res) => {
  const {
    id,
    name,
    email,
    password,
    role,
    permissions,
    companies
  } = req.body;
  const transact = userDb.transaction(() => {
    let userId = id;
    if (id) {
      if (password) {
        userDb.prepare('UPDATE users SET name=?, email=?, password_hash=?, role=? WHERE id=?').run(name, email, bcrypt.hashSync(password, 10), role, id);
      } else {
        userDb.prepare('UPDATE users SET name=?, email=?, role=? WHERE id=?').run(name, email, role, id);
      }
    } else {
      if (!password) throw new Error('Password is required for new users.');
      userId = userDb.prepare('INSERT INTO users (name, email, password_hash, role) VALUES (?,?,?,?)').run(name, email, bcrypt.hashSync(password, 10), role).lastInsertRowid;
    }
    userDb.prepare('DELETE FROM permissions WHERE user_id=?').run(userId);
    userDb.prepare('DELETE FROM user_companies WHERE user_id=?').run(userId);
    const insertPerm = userDb.prepare('INSERT INTO permissions (user_id, permission) VALUES (?,?)');
    (permissions || []).forEach(p => insertPerm.run(userId, p));
    const insertComp = userDb.prepare('INSERT INTO user_companies (user_id, company_id) VALUES (?,?)');
    (companies || []).forEach(c => insertComp.run(userId, c.id));
  });
  try {
    transact();
    res.status(201).json({
      message: 'User saved.'
    });
  } catch (err) {
    res.status(err.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 409 : 500).json({
      message: err.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 'Email already exists.' : err.message
    });
  }
});
router.delete('/users/:id', isAuthenticated, isAdmin, (req, res) => {
  if (req.session.user.id == req.params.id) return res.status(400).json({
    message: 'You cannot delete your own account.'
  });
  try {
    const r = userDb.prepare('DELETE FROM users WHERE id=?').run(req.params.id);
    res.status(r.changes ? 200 : 404).json({
      message: r.changes ? 'User deleted.' : 'User not found.'
    });
  } catch (err) {
    res.status(500).json({
      message: err.message
    });
  }
});
router.get('/companies', isAuthenticated, (req, res) => {
  try {
    res.json(userDb.prepare('SELECT * FROM companies ORDER BY name').all());
  } catch (e) {
    res.status(500).json({
      message: e.message
    });
  }
});
router.post('/companies', isAuthenticated, isAdmin, (req, res) => {
  const {
    id,
    name
  } = req.body;
  try {
    if (id) userDb.prepare('UPDATE companies SET name=? WHERE id=?').run(name, id);else userDb.prepare('INSERT INTO companies (name) VALUES (?)').run(name);
    res.status(201).json({
      message: 'Company saved.'
    });
  } catch (e) {
    res.status(e.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 409 : 500).json({
      message: e.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 'Company name exists.' : e.message
    });
  }
});
router.delete('/companies/:id', isAuthenticated, isAdmin, (req, res) => {
  try {
    const r = userDb.prepare('DELETE FROM companies WHERE id=?').run(req.params.id);
    res.status(r.changes ? 200 : 404).json({
      message: r.changes ? 'Company deleted.' : 'Company not found.'
    });
  } catch (e) {
    res.status(500).json({
      message: e.message
    });
  }
});
router.get('/transporters', isAuthenticated, (req, res) => {
  try {
    res.json(userDb.prepare('SELECT * FROM transporters ORDER BY name').all());
  } catch (e) {
    res.status(500).json({
      message: e.message
    });
  }
});
router.post('/transporters', isAuthenticated, isAdmin, (req, res) => {
  const {
    id,
    name
  } = req.body;
  try {
    if (id) userDb.prepare('UPDATE transporters SET name=? WHERE id=?').run(name, id);else userDb.prepare('INSERT INTO transporters (name) VALUES (?)').run(name);
    res.status(201).json({
      message: 'Transporter saved.'
    });
  } catch (e) {
    res.status(e.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 409 : 500).json({
      message: e.code === 'SQLITE_CONSTRAINT_UNIQUE' ? 'Transporter name exists.' : e.message
    });
  }
});
router.delete('/transporters/:id', isAuthenticated, isAdmin, (req, res) => {
  try {
    const r = userDb.prepare('DELETE FROM transporters WHERE id=?').run(req.params.id);
    res.status(r.changes ? 200 : 404).json({
      message: r.changes ? 'Transporter deleted.' : 'Transporter not found.'
    });
  } catch (e) {
    res.status(500).json({
      message: e.message
    });
  }
});

// --- DISPATCH ORDER ROUTES ---
router.get('/dispatch/orders', isAuthenticated, (req, res) => {
  try {
    const companies = userDb.prepare('SELECT id, name FROM companies').all();
    const transporters = userDb.prepare('SELECT id, name FROM transporters').all();
    const companyMap = new Map(companies.map(c => [c.id, c.name]));
    const transporterMap = new Map(transporters.map(t => [t.id, t.name]));
    const orders = dispatchDb.prepare('SELECT * FROM orders ORDER BY is_urgent DESC, created_at DESC').all();
    res.json(orders.map(order => ({
      ...order,
      company_name: companyMap.get(order.company_id) || 'N/A',
      transporter_name: transporterMap.get(order.transporter_id) || 'N/A'
    })));
  } catch (err) {
    res.status(500).json({
      message: 'Failed to fetch dispatch orders.'
    });
  }
});
router.post('/dispatch/order', isAuthenticated, canDispatch, (req, res) => {
  const {
    sr_no,
    party_name,
    place,
    district,
    quantity_kg,
    company_id,
    transporter_id,
    is_urgent,
    is_on_hold
  } = req.body;
  if (!party_name || !place || !quantity_kg || !company_id || !transporter_id) return res.status(400).json({
    message: 'Missing required fields.'
  });
  try {
    if (!userDb.prepare('SELECT id FROM companies WHERE id = ?').get(company_id)) return res.status(400).json({
      message: 'Invalid Company selected.'
    });
    if (!userDb.prepare('SELECT id FROM transporters WHERE id = ?').get(transporter_id)) return res.status(400).json({
      message: 'Invalid Transporter selected.'
    });
    const result = dispatchDb.prepare(`INSERT INTO orders (sr_no, party_name, place, district, quantity_kg, company_id, transporter_id, is_urgent, is_on_hold, created_by_id, created_by_name) VALUES (?,?,?,?,?,?,?,?,?,?,?)`).run(sr_no || null, party_name, place, district || null, parseFloat(quantity_kg), parseInt(company_id), parseInt(transporter_id), is_urgent ? 1 : 0, is_on_hold ? 1 : 0, req.session.user.id, req.session.user.name);
    res.status(201).json({
      id: result.lastInsertRowid
    });
  } catch (err) {
    res.status(500).json({
      message: "Failed to save order. " + err.message
    });
  }
});
router.post('/dispatch/orders/update-status', isAuthenticated, canUpdateDispatch, (req, res) => {
  const {
    updates
  } = req.body;
  if (!updates || !Array.isArray(updates)) return res.status(400).json({
    message: 'Invalid request body.'
  });
  const stmt = dispatchDb.prepare('UPDATE orders SET status = ?, remark = ?, is_on_hold = ?, hold_reason = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?');
  const transact = dispatchDb.transaction(updateList => {
    for (const update of updateList) {
      const isOnHold = update.status === 'On Hold' ? 1 : 0;
      const holdReason = isOnHold ? update.hold_reason : null;
      stmt.run(update.status, update.remark, isOnHold, holdReason, update.id);
    }
  });
  try {
    transact(updates);
    res.status(200).json({
      message: 'Statuses updated successfully.'
    });
  } catch (err) {
    res.status(500).json({
      message: 'Failed to update statuses. ' + err.message
    });
  }
});
router.put('/dispatch/order/:id/remove-hold', isAuthenticated, (req, res) => {
  const {
    id
  } = req.params;
  const user = req.session.user;
  if (user.role !== 'admin' && user.role !== 'manager') {
    return res.status(403).json({
      message: "You don't have permission to remove holds."
    });
  }
  try {
    const result = dispatchDb.prepare(`UPDATE orders SET is_on_hold = 0, hold_reason = NULL, status = 'Open', updated_at = CURRENT_TIMESTAMP WHERE id = ?`).run(id);
    res.status(result.changes > 0 ? 200 : 404).json({
      message: result.changes > 0 ? 'Hold removed.' : 'Order not found.'
    });
  } catch (err) {
    res.status(500).json({
      message: 'Failed to remove hold. ' + err.message
    });
  }
});
router.put('/dispatch/order/:id', isAuthenticated, (req, res) => {
  const {
    id
  } = req.params;
  const {
    status
  } = req.body;
  const user = req.session.user;
  if (user.role !== 'admin' && user.role !== 'manager') {
    return res.status(403).json({
      message: "You don't have permission to perform this action."
    });
  }
  if (status !== 'Open') {
    return res.status(400).json({
      message: 'Can only revert a dispatched order back to "Open".'
    });
  }
  try {
    const result = dispatchDb.prepare('UPDATE orders SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ? and status = "Dispatched"').run(status, id);
    res.status(result.changes > 0 ? 200 : 404).json({
      message: result.changes > 0 ? 'Order status reverted.' : 'Order not found or was not dispatched.'
    });
  } catch (err) {
    res.status(500).json({
      message: 'Failed to update order. ' + err.message
    });
  }
});
router.delete('/dispatch/order/:id', isAuthenticated, isAdmin, (req, res) => {
  try {
    const result = dispatchDb.prepare('DELETE FROM orders WHERE id = ?').run(req.params.id);
    res.status(result.changes > 0 ? 200 : 404).json({
      message: result.changes > 0 ? 'Order deleted.' : 'Order not found.'
    });
  } catch (err) {
    res.status(500).json({
      message: err.message
    });
  }
});
module.exports = router;